using System;
using System.Text;
using Microsoft.Uddi;
using Microsoft.Uddi.Api;

namespace com.swvoyages.reservation {

	///
	/// <summary>La classe <code>ReservationFinder</code> implmente 
	/// l'interface <code>ReservationLookup</code>. Elle est utilise 
	/// pour dterminer la manire de rechercher l'une des implmentations 
	/// du service type de rservation dont le nom est pass en paramtre, 
	/// dans le registre UDDI dont l'URL de l'API de recherche est galement 
	/// fournie en paramtre.</summary>
	///
	public class ReservationFinder {

		private static ReservationFinder uniqueInstance;
		private static Random randomizer = new Random();

		///
		/// <summary>Construit une instance de la classe <code>ReservationFinder</code>.
		/// </summary>
		/// 
		private ReservationFinder() {
		}

		///
		/// <summary>Renvoie le singleton de la classe <code>ReservationFinder</code>.
		/// </summary>
		/// 
		/// <returns>le singleton de la classe<code>ReservationFinder</code>.
		/// </returns>
		///
		public static ReservationFinder getUniqueInstance() {
			if (uniqueInstance == null) {
				uniqueInstance = new ReservationFinder();
			}
			return uniqueInstance;
		}

		///
		/// <summary>Renvoie le tableau des coordonnes de rservation 
		/// du partenaire trouv. Celui-ci doit implmenter le service
		/// type dont le nom est pass en paramtre et ses coordonnes
		/// sont recherches dans le registre UDDI dont l'URL de l'API 
		/// de recherche est communique.</summary>
		///
		/// <param name="inquireURL">URL de l'API de recherche du
		///	registre UDDI slectionn</param>
		/// <param name="tModelName">nom du service type dont une
		///	implmentation est recherche</param>
		///
		/// <returns>String[0] URL du fichier WSDL de description du
		/// service type implment - String[1] URL du point d'accs 
		/// au service du partenaire</returns>
		///
		/// <exception cref="ReservationException">lance si une anomalie 
		/// se produit durant la recherche d'un partenaire</exception>
		///
		public String [] lookup(String inquireURL, String tModelName) {
 
			try {
				Inquire.Url = inquireURL;

				// Recherche des services type dont le nom est spcifi.
				FindTModel ft = new FindTModel();
				ft.Name = tModelName;
				TModelList tList = ft.Send();
				if (tList.TModelInfos.Count == 0) {
					String message = "pas de service(s) type " + tModelName + " trouv(s)";
					throw new ReservationException(message);
				}

				// Slectionne au hasard l'un des services type trouvs. 
				int itmi = randomizer.Next(tList.TModelInfos.Count);
				String tModelKey = tList.TModelInfos[itmi].TModelKey;

				// Recherche des entits mtier qui implmentent le service type.
				StringCollection tModelKeys = new StringCollection();
				tModelKeys.Add(tModelKey);
				FindBusiness fb = new FindBusiness();
				fb.TModelKeys = tModelKeys;
				BusinessList blst = fb.Send();
				if (blst.BusinessInfos.Count==0) {
					String message = "aucune entit mtier n'implmente le service type " + tModelKey;
					throw new ReservationException(message);
				}
	    
				// Slectionne au hasard l'une des entits mtier trouves. 
				int ibi = randomizer.Next(blst.BusinessInfos.Count);
				String businessKey = blst.BusinessInfos[ibi].BusinessKey;

				// Recherche des services mtier qui implmentent le service type.
				FindService fs = new FindService();
				fs.TModelKeys = tModelKeys;
				fs.BusinessKey = businessKey;
				ServiceList slst = fs.Send();
				if (slst.ServiceInfos.Count==0) {
					String message = "aucun service mtier n'implmente le service type " + tModelKey;
					throw new ReservationException(message);
				}

				// Slectionne au hasard l'un des services mtier trouvs. 
				int isi = randomizer.Next(slst.ServiceInfos.Count);
				String serviceKey = slst.ServiceInfos[ibi].ServiceKey;

				// Recherche des liaisons qui implmentent le service type.
				FindBinding fbt = new FindBinding();
				fbt.TModelKeys = tModelKeys;
				fbt.ServiceKey = serviceKey;
				BindingDetail bd = fbt.Send();
				if (bd.BindingTemplates.Count==0) {
					String message = "aucune liaison n'implmente le service type " + tModelKey;
					throw new ReservationException(message);
				}

				// Slectionne au hasard l'une des liaisons trouves. 
				int ibt = randomizer.Next(bd.BindingTemplates.Count);

				// Rcupre l'URL du point d'accs au service slectionn.
				String address = bd.BindingTemplates[ibt].AccessPoint.Text;

				// Rcupre l'URL du modle WSDL du service slectionn.
				GetTModelDetail gt = new GetTModelDetail();
				gt.TModelKeys.Add(tModelKey);
				TModelDetail td = gt.Send();
				String wsdlURL = td.TModels[0].OverviewDoc.OverviewURL;

				return new String[]{wsdlURL, address};
			}
			catch (Exception e) {
				StringBuilder sb = new StringBuilder();
				sb.Append("\rune ");
				sb.Append(e.GetType().FullName);
				sb.Append(" s'est produite durant la recherche d'un partenaire de rservation");
				if ((e.Message!=null) && (!(e.Message.Equals("")))) {
					sb.Append(" : ");
					sb.Append(e.Message);
				}
				throw new ReservationException(sb.ToString(), e);
			}
		}
	}
}
